//=========================================================================
// network.cpp
// Example implementation of the network-layer code
/*=========================================================================
Copyright (C) 2013-2014 Giorgio Biagetti.

Distributed under the Boost Software License, Version 1.0.

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

=========================================================================*/

/* Usage notes:
 * To convert this file from C++ to plain C,
 * pipe it through the following sed expression:
 * "s-[A-Za-z_][A-Za-z0-9_]*::--g"
 */

#ifndef __cplusplus
#define __NETWORK_C
#define NWK_DISABLE_SEND_REMOTE_CONFIG
#endif

#include "network.h"
#include "errors.h"

#include <string.h>

#ifdef NWK_DEBUG_DUMP
#include <stdio.h>
#define __STDC_FORMAT_MACROS
#include <inttypes.h>
#define NWK_DEBUG_LOG(msg) printf(" ** %012" PRIX64 ":   %s!\n", nwk_station_address, msg)
#else
#define NWK_DEBUG_LOG(msg)
#endif

#ifdef STDOUT_REDIRECT
int custom_printf (const char *fmt, ...);
#define printf custom_printf
#endif


// Implementation of the Forward Information Base (FIB) class:

void fib_flush ()
{
	fib_size = 0;
}

bool fib_add (uint16_t net, uint8_t length, char device, int direction)
{
	// Check if there is room to add an entry:
	if (!(fib_size < fib_max_entries)) return false;

	// keep the table ordered by mask_length:
	unsigned i = 0;
	while (i < fib_size && length <= fib_table[i].mask_length) ++i;
	// make room for the new entry if necessary:
	if (i < fib_size)
		memmove(fib_table + i + 1, fib_table + i, (fib_size - i) * sizeof (struct fib_entry));
	++fib_size;

	// insert the new entry:
	fib_table[i].net = net;
	fib_table[i].mask = 0x10000 - (1 << (16 - length));
	fib_table[i].mask_length = length;
	fib_table[i].direction = direction;
	fib_table[i].type = 0;
	fib_table[i].device = device;

	return true;
}

bool fib_add_span (uint16_t first, uint16_t last, char device, int direction)
{
	// Check if there is room to add an entry:
	if (!(fib_size < fib_max_entries)) return false;

	// keep the table ordered by mask_length:
	uint8_t length = 15; // conventional placement of span-based routes.
	unsigned i = 0;
	while (i < fib_size && length <= fib_table[i].mask_length) ++i;
	// make room for the new entry if necessary:
	if (i < fib_size)
		memmove(fib_table + i + 1, fib_table + i, (fib_size - i) * sizeof (struct fib_entry));
	++fib_size;

	// insert the new entry:
	fib_table[i].net = first;
	fib_table[i].mask = last;
	fib_table[i].mask_length = length;
	fib_table[i].direction = direction;
	fib_table[i].type = 1;
	fib_table[i].device = device;

	return true;
}

bool fib_del (uint16_t net, uint8_t length)
{
	uint16_t mask = 0x10000 - (1 << (16 - length));
	unsigned old_size = fib_size;
	unsigned i = 0;
	while (i < fib_size) {
		// since the table is ordered, we can stop looking as soon as we find a shorter mask:
		if (fib_table[i].mask_length < length) break;
		if (fib_table[i].type == 0 && (fib_table[i].net & mask) == net) {
			// the current entry matches the net to be removed, so remove it and shorten the table:
			if (i < --fib_size)
				memmove(fib_table + i, fib_table + i + 1, (fib_size - i) * sizeof (struct fib_entry));
		} else {
			// the current entry does not match, keep looking...
			++i;
		}
	}

	// return true if any matching entry was found:
	return fib_size != old_size;
}

bool fib_del_span (uint16_t first, uint16_t last)
{
	unsigned old_size = fib_size;
	unsigned i = 0;
	while (i < fib_size) {
		// since the table is ordered, we can stop looking as soon as we find a shorter mask:
		if (fib_table[i].mask_length < 15) break;
		if (fib_table[i].type == 1 && fib_table[i].net >= first && fib_table[i].mask <= last) {
			// the current entry matches the net to be removed, so remove it and shorten the table:
			if (i < --fib_size)
				memmove(fib_table + i, fib_table + i + 1, (fib_size - i) * sizeof (struct fib_entry));
		} else {
			// the current entry does not match, keep looking...
			++i;
		}
	}

	// return true if any matching entry was found:
	return fib_size != old_size;
}

struct fib_entry const *fib_lookup (uint16_t address)
{
	unsigned i;
	for (i = 0; i < fib_size; ++i)
		if (
			(fib_table[i].type == 0 && fib_table[i].net == (address & fib_table[i].mask))
		||
			(fib_table[i].type == 1 && address >= fib_table[i].net && address <= fib_table[i].mask)
		) {
			return fib_table + i;
		}
	return NULL;
}

#ifdef NWK_DEBUG_DUMP
void fib_dump () const
{
	for (unsigned i = 0; i < fib_size; ++i) {
		if (fib_table[i].type == 0) {
			printf("  %04X/%02X (%04X) %c %c\n", 
				fib_table[i].net,
				fib_table[i].mask_length,
				fib_table[i].mask,
				fib_table[i].direction ? fib_table[i].direction > 0 ? '+' : '-' : '=',
				fib_table[i].device ? fib_table[i].device : '0'
			);
		} else if (fib_table[i].type == 1) {
			printf("  %04X-%04X      %c %c\n", 
				fib_table[i].net,
				fib_table[i].mask,
				fib_table[i].direction ? fib_table[i].direction > 0 ? '+' : '-' : '=',
				fib_table[i].device ? fib_table[i].device : '0'
			);
		}
	}
}
#endif


// Implementation of the network layer:

struct header
{
	unsigned len;             // length in bytes of the encoded packet header (not of this struct!)
	uint8_t  seq         : 4; // sequence number
	uint8_t  code        : 3; // command code
	uint8_t  target      : 1; // 0=network, 1=transport
	uint8_t  flags       : 2; // address coding: 00=SRC, 01=SRC+DST, 10=DST, 11=DST+MAC
	uint8_t  mac_address : 1; // MAC address specified (i.e., flags == 3);
	uint8_t  reserved    : 5; // just for padding...
	int16_t  direction;       // +1=towards leaves, -1=towards root
	uint16_t hops;            // current number of hops from root
	uint16_t src;             // source network address
	uint16_t dst;             // destination network address
	uint16_t distance;        // destination hop count,   if (flags == 3)
	uint64_t mac;             // destination MAC address, if (flags == 3). Only 48 bits are used, right justified.
};

#ifdef NWK_DEBUG_DUMP
const char opcode_names[2][8] = { { 'A', 'N', 'G', 'T', 'M', 'P', 'S', 'C' }, { 'A', 'N', 'G', 'g', 'M', 'm', 'S', 's' } };
void dump (const char *prefix, uint64_t node_address) const
{
	uint8_t *data = (uint8_t *) this->data;
	if (!data || length < 2) {
		printf("WRONG PACKET!\n");
		return;
	}
	// extract header fields:
	struct header h;
	h.len = 2;
	h.code      = (data[0] >> 4) & 0x07;
	h.seq       =  data[0] & 0x0F;
	h.target    = (data[0] >> 7) & 0x01;
	h.flags     =  data[1] >> 6;
	h.direction = (data[1] & 0x20) ? -1 : +1;
	h.hops      = (data[1] & 0x1F) == 0x1F ? (h.len += 2), (((uint16_t) data[h.len - 2] << 8) | data[h.len - 1]) : data[1] & 0x1F;
	h.src       = (data[1] & 0x80) == 0x00 ? (h.len += 2), (((uint16_t) data[h.len - 2] << 8) | data[h.len - 1]) : 0x0000;
	h.dst       = (data[1] & 0xC0) != 0x00 ? (h.len += 2), (((uint16_t) data[h.len - 2] << 8) | data[h.len - 1]) : 0x0000;
	h.distance  = (data[1] & 0xC0) == 0xC0 ? (h.len += 8), (((uint16_t) data[h.len - 2] << 8) | data[h.len - 1]) : 0x0000;
	h.mac       = (data[1] & 0xC0) == 0xC0 ? (
		((uint64_t) data[h.len - 8] << 40) |
		((uint64_t) data[h.len - 7] << 32) |
		((uint64_t) data[h.len - 6] << 24) |
		((uint64_t) data[h.len - 5] << 16) |
		((uint64_t) data[h.len - 4] <<  8) |
		((uint64_t) data[h.len - 3] <<  0)
	) : 0x000000000000;
	printf("%s %012" PRIX64 "/%c ", prefix, node_address, device);
	if (h.flags)
		printf(" D=%04X", h.dst);
	else
		printf(" D=----");
	if ((h.flags & 2) == 0)
		printf(" S=%04X", h.src);
	else
		printf(" S=----");
	if (h.flags == 3)
		printf(" M=%012" PRIX64 "@%04X", h.mac, h.distance);
	else
		printf(" M=------------@----");
	printf(" H=%04X%c T=%d C=%c S=%01X :", h.hops, h.direction ? h.direction > 0 ? '+' : '-' : '=', h.target, opcode_names[h.target][h.code], h.seq);
	while (h.len < length)
		printf(" %02X", data[h.len++]);
	printf("\n");
}
#endif

// Utility functions for tracing and encoding
#ifdef __cplusplus
#include <math.h>

// Base of logarithmic encoding: log_base = pow(2, (32 - 4) / (256 - 16))
// should ensure a maximum relative error smaller than 4.05%
static const double log_base = 1.0842268703014184;

static uint8_t log_encode (uint32_t x)
{
	if (x < 16U) return (uint8_t) x;
	return (uint8_t) 16 + (uint8_t) floor((log2(x) - 4) / log2(log_base));
}

static uint32_t log_decode (uint8_t n)
{
	if (n < 16U) return n;
	double x = 16 * pow(log_base, n - 16) * (log_base + 1) / 2;
	// WARNING: relative error is guaranteed against floating-point value x,
	// if x is quantized, the error can go up of a few per cent for small x.
	return (uint32_t) (x + 0.5);
}

#endif

#ifdef __cplusplus
router (uint64_t mac)
{
	nwk_init(mac);
}

uint64_t address () const
{
	return nwk_station_address;
}
#endif

void nwk_init (uint64_t mac)
{
	fib_flush();
	nwk_station_address = mac;
	nwk_depth_level = 0;
	outgoing_packet.length = 0;
	outgoing_packet.data = NULL;
	memset(connections, 0, sizeof connections);
	nwk_hash_clear();
	netevent_received = NULL;
	datagram_received = NULL;
	netreply_received = NULL;
	options.enable_incoming = 1;
	options.enable_forward  = 1;
	options.enable_ping     = 1;
	options.enable_config   = 1;
}

int nwk_connect_device (char device, send_packet_t send_packet)
{
	unsigned hash = (uint8_t) device % max_interfaces;
	if (connections[hash].device) return -ENOMEM;
	connections[hash].device = device;
	connections[hash].send_packet = send_packet;
	return 0;
}

void nwk_hash_clear (void)
{
	nwk_tx_sequence_counter = 0xFF;   // an invalid value
	nwk_rx_sequence_counter = 0xFF;   // an invalid value
	nwk_rx_sequence_source  = 0xFFFF; // broadcast address can't ever be used as source
	memset(dst_hash_table, 0, sizeof dst_hash_table);
	memset(seq_hash_table, 0, sizeof seq_hash_table);
}

int nwk_hash_address (uint16_t address)
{
	// TODO: hash table entries must time out after a while, otherwise table will fill up!
	uint16_t key = ~address, hash = address % dst_hash_table_size, x = hash;
	while (dst_hash_table[x] && dst_hash_table[x] != key) {
		x = (x + 1) % dst_hash_table_size;
		if (x == hash) return -ENOMEM;
	}
	dst_hash_table[x] = key;
	return x;
}

int send_packet (struct packet const *p)
{
	unsigned hash = (uint8_t) p->device % max_interfaces;
	if (!connections[hash].device) return -ENODEV;
	if (connections[hash].device != p->device) return -ENODEV;
	int status = connections[hash].send_packet(p);
	return status < 0 ? status : nwk_tx_sequence_counter;
}



#define SET_CODE(ptr, code) (*(ptr) = (*(ptr) & 0x8F) | ((code) << 4))



void packet_received (struct packet const *p)
{
	// check parameters:
	uint8_t *data = p ? (uint8_t *) p->data : NULL;
	if (!data || p->length < 2) return;
	// extract header fields:
	// WARNING: the following code might access up to 14 bytes of memory from data. Make sure it is _always_ allocated!
	struct header h;
	h.len       = 2;
	h.code      = (data[0] >> 4) & 0x07;
	h.seq       =  data[0] & 0x0F;
	h.target    = (data[0] >> 7) & 0x01;
	h.flags     =  data[1] >> 6;
	h.direction = (data[1] & 0x20) ? -1 : +1;
	h.hops      = (data[1] & 0x1F) == 0x1F ? (h.len += 2), (((uint16_t) data[h.len - 2] << 8) | data[h.len - 1]) : data[1] & 0x1F;
	h.src       = (data[1] & 0x80) == 0x00 ? (h.len += 2), (((uint16_t) data[h.len - 2] << 8) | data[h.len - 1]) : 0x0000;
	h.dst       = (data[1] & 0xC0) != 0x00 ? (h.len += 2), (((uint16_t) data[h.len - 2] << 8) | data[h.len - 1]) : 0x0000;
	h.distance  = (data[1] & 0xC0) == 0xC0 ? (h.len += 8), (((uint16_t) data[h.len - 2] << 8) | data[h.len - 1]) : 0x0000;
	h.mac       = (data[1] & 0xC0) == 0xC0 ? (h.mac_address = 1), (
		((uint64_t) data[h.len - 8] << 40) |
		((uint64_t) data[h.len - 7] << 32) |
		((uint64_t) data[h.len - 6] << 24) |
		((uint64_t) data[h.len - 5] << 16) |
		((uint64_t) data[h.len - 4] <<  8) |
		((uint64_t) data[h.len - 3] <<  0)
	) : (h.mac_address = 0);
	// check packet length:
	if (p->length < h.len) return;
	// determine packet fate:
	if (h.mac_address && h.mac == nwk_station_address) {
		// MAC address match. Check hop count:
		if (h.hops != h.distance) return;
	} else {
		// MAC address mismatch...
		// check routing level:
		if (h.hops != nwk_depth_level) return;
		// check source address and ingress port:
		struct fib_entry const *src_route = fib_lookup(h.src);
		// drop packet if ingress port is wrong:
		if (!src_route || src_route->device != p->device) return;
		// ckeck destination address:
		struct fib_entry const *dst_route = fib_lookup(h.dst);
		// Drop the packet if there is no known destination route:
		if (!dst_route) return;
		// Can't forward packets from generic route back, so drop them if it's the case:
		if ((src_route->direction < 0) && (dst_route->direction < 0)) return;
		if ((src_route->direction < 0) == (h.direction < 0)) return;
		
		// At this point, we accepted the packet, either to be forwarded or processed locally.

#ifndef __cplusplus
		// Special processing for TRACE command, only done in firmware
		if (h.code == CODE_TRACE) {
			uint8_t len = p->length - h.len;
			if (len >= 6) {
				uint8_t *payload = data + h.len;
				uint16_t min_hops = ((uint16_t) payload[0] << 8) + payload[1];
				uint16_t max_hops = (len - 6) / 8 + min_hops;
				if (nwk_depth_level >= min_hops && (nwk_depth_level < max_hops || (nwk_depth_level == max_hops && h.direction > 0))) {
					payload += 2 + (nwk_depth_level - min_hops) * 8 + (h.direction < 0 ? 4 : 0);
					payload[0] = 0xAA;
					payload[1] = h.direction;
					payload[2] = p->signal_strength;
					payload[3] = p->signal_quality;
				}
			}
		}
#endif

		// Check if we need to forward the packet over a specific interface:
		if (dst_route->device) {
			if (!options.enable_forward) return;
			struct packet forwarding = {dst_route->device};
			forwarding.length = p->length;
			forwarding.data   = p->data;
			// We should adjust the direction and the hop count.
			h.direction = dst_route->direction;
			if (h.direction < 0 && h.hops) --h.hops;
			if (h.direction > 0 && !++h.hops) return;
			// We might also need to change the length as the new hop count might occupy a different number of bits!
			if (h.direction > 0 && h.hops == 31) {
				forwarding.length += 2;
				memmove(data + 4, data + 2, p->length - 2);
			}
			if (h.direction < 0 && h.hops == 30) {
				forwarding.length -= 2;
				memmove(data + 2, data + 4, p->length - 4);
			}
			// Store new direction:
			if (h.direction < 0)
				data[1] |=  0x20;
			else
				data[1] &= ~0x20;
			// Store new hop count:
			if (h.hops < 31) {
				data[1] &= ~0x1F;
				data[1] |= h.hops;
			} else {
				data[1] |=  0x1F;
				data[2] = (uint8_t) (h.hops >> 8);
				data[3] = (uint8_t) (h.hops & 0xFF);
			}
			// Forward the packet.
			NWK_DEBUG_LOG("forwarding packet");
			send_packet(&forwarding);
			return;
		}
	}

	// if we ended up here, the packet ought to be accepted locally
	data += h.len;
	uint8_t len = p->length - h.len;

	if (h.target) {
		if (!options.enable_incoming) return;
		// check sequence number to identify duplicated commands
		// TODO: nwk_rx_sequence_counter must time out after a while!
		if (h.src == nwk_rx_sequence_source && h.seq == nwk_rx_sequence_counter) {
			// check if a reply had already been sent for that command,
			// send it again if found, ignore incoming packet otherwise:
			if (h.code == CODE_ACK || h.code == CODE_NACK) return;
			if (outgoing_packet.data && (*(uint8_t const *) outgoing_packet.data & 0x0F) == nwk_rx_sequence_counter) {
				NWK_DEBUG_LOG("duplicated packet: repeating reply");
				send_packet(&outgoing_packet);
			} else {
				NWK_DEBUG_LOG("duplicated packet: no previous reply");
			}
			return;
		}
	}
	// a valid and previously unseen packet has been received: update sequence information
	nwk_rx_sequence_source  = h.src;  // remember last seen packet source address
	nwk_rx_sequence_counter = h.seq;  // remember last seen packet sequence
	outgoing_packet.data = NULL;      // erase the old reply buffer
	// update sequence number table:
	int x = nwk_hash_address(h.src);
	if (x < 0) return;
	seq_hash_table[x] = h.seq + 1;
	if (h.target) {
		// send the packet to the transport layer:
		NWK_DEBUG_LOG("datagram received");
		if (datagram_received) datagram_received(h.src, h.seq, h.code, data, len);
	} else {
		// process the packet here at the network layer.
		// we also accept duplicated packets, be sure that repeating an action does not have side effects!
		if (h.code == CODE_ACK || h.code == CODE_NACK || (h.code == CODE_TRACE && h.direction < 0)) {
			// network-level reply received.
			// send to application for further processing...
			NWK_DEBUG_LOG("reply received");
			if (netreply_received) netreply_received(h.src, h.seq, h.code, data, len);
			return;
		}
		// now we must process the command and send back a reply.
		// Fix flags & hop count, switch src and dst flags:
		h.len = 2;
		h.flags = h.src ? h.dst ? 1 : 2 : 0;
		h.direction = -h.direction;
		uint16_t hops = h.hops;
		if (h.direction < 0 || !h.dst) {
			if (hops) --hops;
		} else if (h.direction > 0) {
			if (!++hops) return;
		}
		// Prepare and encode the reply header in-place.
		// since it's a reply, the header can only shrink,
		// so there is no risk of overwriting the payload,
		// and the sequence number is automatically copied.
		uint8_t *buf = (uint8_t *) p->data;
		buf[1] = (h.flags << 6) | (h.direction < 0 ? 0x20 : 0x00) | (hops < 31 ? hops : 0x1F);
		if (hops >= 31) {
			buf[h.len++] = (uint8_t) (hops >> 8);
			buf[h.len++] = (uint8_t) (hops & 0xFF);
		}
		if ((h.flags & 2) == 0) {
			buf[h.len++] = (uint8_t) (h.dst >> 8);
			buf[h.len++] = (uint8_t) (h.dst & 0xFF);
		}
		if (h.flags) {
			buf[h.len++] = (uint8_t) (h.src >> 8);
			buf[h.len++] = (uint8_t) (h.src & 0xFF);
		}
		struct packet reply = {p->device};
		reply.length = h.len;
		reply.data   = buf;
		// Now interpret and execute the command:
		switch (h.code) {
		case CODE_CONFIG:
			if (!options.enable_config) return;
			NWK_DEBUG_LOG("config received");
			// store previous network status in reserved area:
			h.reserved = !fib_size;
			if (h.mac_address) {
				// if the MAC address was specified, we reset the configuration and set the network address.
				fib_flush();
				nwk_hash_clear();
				// add local address:
				fib_add(h.dst, 16, '\0', 0);
				// add backward route:
				fib_add(0x0000,  0, p->device, -1);
				// set router depth:
				nwk_depth_level = h.hops;
			}
			// dissect payload and add specified routes:
			h.len = 0; // count correctly processed bytes here.
			while (len >= 4) {
				char destination = data[0];
				uint8_t length = data[1];
				bool done = false;
				if (length > 16) {
					// span-type route:
					uint16_t span;
					if (length == 0xFF) {
						data += 2;
						len -= 2;
						if (len < 4) break;
						span = ((uint16_t) data[0] << 8) | data[1];
					} else {
						span = length - 16;
					}
					uint16_t first = ((uint16_t) data[2] << 8) | data[3];
					uint16_t last = first + span;
					if (last < first) break;
					// update FIB:
					done = fib_del_span(first, last);
					if (destination < 0x7F) // 0x7F is the DELete command
						done = fib_add_span(first, last, destination, !!destination);
				} else {
					// mask-type route:
					uint16_t net = ((uint16_t) data[2] << 8) | data[3];
					// update FIB:
					done = fib_del(net, length);
					if (destination < 0x7F) // 0x7F is the DELete command
						done = fib_add(net, length, destination, !!destination);
				}
				if (!done) break;
				// go to next command:
				data += 4;
				len -= 4;
				h.len += 4;
			}
			if (len) {
				SET_CODE(buf, CODE_NACK);
				buf[reply.length++] = EINVAL;
				buf[reply.length++] = h.len; // return length of processed data.
			} else {
				SET_CODE(buf, CODE_ACK);
			}
			// notify application of what has just happened:
			if (netevent_received) {
				netevent_received(fib_size ? h.reserved ? nwk_network_up : nwk_network_reconf : nwk_network_down);
			}
#ifdef NWK_DEBUG_DUMP
			fib_dump();
#endif
			break;
		case CODE_PING:
			if (!options.enable_ping) return;
			NWK_DEBUG_LOG("ping received");
			SET_CODE(buf, CODE_ACK);
			// copy ping message payload:
			memmove(buf + reply.length, data, len);
			reply.length += len;
			break;
		case CODE_TRACE:
			if (!options.enable_ping) return;
			NWK_DEBUG_LOG("trace received");
			// code must remain TRACE
			// copy message payload:
			memmove(buf + reply.length, data, len);
			reply.length += len;
			break;
		default:
			return;
		}
		send_packet(&reply);
	}
}


// packet creation routines:

int create_packet (bool target, uint16_t dst, uint16_t hop, uint64_t mac, uint8_t code, const void *data, size_t length)
{
	// Determine source address from routing table:
	if (!fib_size) return -ENONET;
	uint16_t src = fib_table[0].net;
	if (src && mac) return -EINVAL;
	// Determine output route:
	struct fib_entry const *dst_route = fib_lookup(dst);
	if (!dst_route) return -ENOWAY;
	// Build packet metadata & header:
	struct packet *p = &outgoing_packet;
	uint8_t *buf = packet_buffer;
	p->device = dst_route->device;
	p->flags = 0;
	p->signal_strength = 0;
	p->signal_quality = 0;
	p->timestamp = 0;
	p->data = buf;
	if (code <= CODE_NACK) {
		// send replies back with the same sequence number
		nwk_tx_sequence_counter = nwk_rx_sequence_counter;
	} else {
		// this is not a reply, determine a new suitable sequence number for the destination:
		int x = nwk_hash_address(dst);
		if (x < 0) return x;
		nwk_tx_sequence_counter = seq_hash_table[x]++ % 0x10;
	}
	struct header h = {
		2,       /* basic header length           */
		(uint8_t) nwk_tx_sequence_counter,
		code,
		target,  /* target layer                  */
		0, 0, 0, /* flags (will be fixed later)   */
		dst_route->direction,
		nwk_depth_level,
		src, dst /* network addresses             */
	};
	// Fix flags & hop count:
	if (dst) h.flags = src ? 1 : 2;
	if (mac) h.flags = 3;
	if (dst_route->direction < 0 && h.hops) --h.hops;
	if (dst_route->direction > 0 && !++h.hops) return -ERANGE;
	// Encode header:
	buf[0] = (h.target << 7) | (h.code << 4) | h.seq;
	buf[1] = (h.flags << 6) | (h.direction < 0 ? 0x20 : 0x00) | (h.hops < 31 ? h.hops : 0x1F);
	if (h.hops >= 31) {
		buf[h.len++] = (uint8_t) (h.hops >> 8);
		buf[h.len++] = (uint8_t) (h.hops & 0xFF);
	}
	if ((h.flags & 2) == 0) {
		buf[h.len++] = (uint8_t) (src >> 8);
		buf[h.len++] = (uint8_t) (src & 0xFF);
	}
	if (h.flags) {
		buf[h.len++] = (uint8_t) (dst >> 8);
		buf[h.len++] = (uint8_t) (dst & 0xFF);
	}
	if (h.flags == 3) {
		buf[h.len++] = (uint8_t) ((mac >> 40) & 0xFF);
		buf[h.len++] = (uint8_t) ((mac >> 32) & 0xFF);
		buf[h.len++] = (uint8_t) ((mac >> 24) & 0xFF);
		buf[h.len++] = (uint8_t) ((mac >> 16) & 0xFF);
		buf[h.len++] = (uint8_t) ((mac >>  8) & 0xFF);
		buf[h.len++] = (uint8_t) ((mac >>  0) & 0xFF);
		buf[h.len++] = (uint8_t) ((hop >>  8) & 0xFF);
		buf[h.len++] = (uint8_t) ((hop >>  0) & 0xFF);
	}
	// Store header size:
	p->length = h.len;
	// If data is already known, add it to the packet
	if (data) {
		if (length > max_packet_length - p->length) return -EMSGSIZE;
		memcpy(buf + p->length, data, length);
		p->length += length;
	}
	// That's all!
	return 0;
}

/*	Sends a datagram to the specified network address.
 *
 *	Possible errors:
 *	-ENONET       : network layer not yet configured
 *	-ENOWAY       : no known route exists towards the destination address
 *	-ENODEV       : the route to the destination address specifies an unavailable device
 *	-ERANGE       : the route to the destination address digs too deep into the tree...
 *	-EMSGSIZE     : data too long to fit in the packet payload
 */
int send_datagram (uint16_t dst_address, const void *data, size_t length, uint8_t code)
{
	int status = create_packet(true, dst_address, 0, 0, code, data, length);
	if (status < 0) return status;
	return send_packet(&outgoing_packet);
}

int send_nwk (uint16_t dst_address, const void *data, size_t length, uint8_t code)
{
	int status = create_packet(false, dst_address, 0, 0, code, data, length);
	if (status < 0) return status;
	return send_packet(&outgoing_packet);
}

int send_ping (uint16_t dst_address, const void *data, size_t length)
{
	int status = create_packet(false, dst_address, 0, 0, CODE_PING, data, length);
	if (status < 0) return status;
	return send_packet(&outgoing_packet);
}

int send_trace (uint16_t dst_address, uint16_t start_hop, uint16_t max_hops)
{
	int status = create_packet(false, dst_address, 0, 0, CODE_TRACE, NULL, 0);
	if (status < 0) return status;
	uint8_t *buf = (uint8_t *) outgoing_packet.data;
	uint32_t length = 2 + (max_hops + 1) * 8 - 4;
	if (length > max_packet_length - outgoing_packet.length) return -EMSGSIZE;
	buf[outgoing_packet.length++] = start_hop >> 8;
	buf[outgoing_packet.length++] = start_hop & 0xFF;
	length -= 2;
	memset(buf + outgoing_packet.length, 0, length);
	outgoing_packet.length += length;
	return send_packet(&outgoing_packet);
}


int nwk_config_prepare (uint16_t dst_address, uint64_t mac_address, uint16_t distance)
{
#ifdef NWK_DISABLE_SEND_REMOTE_CONFIG
	(void) dst_address;
	(void) mac_address;
	(void) distance;
	return -ENOSYS;
#else
	// reset sequence counter for the node being configured:
	int x = nwk_hash_address(dst_address);
	if (x < 0) return x;
	seq_hash_table[x] = 0;
	// prepare packet header:
	int status = create_packet(false, dst_address, distance, mac_address, CODE_CONFIG, NULL, 0);
	if (status < 0) return status;
	return 0;
#endif
}

int nwk_packet_size (void)
{
	return outgoing_packet.data ? outgoing_packet.length : 0;
}


int nwk_send_buffer (void)
{
	return send_packet(&outgoing_packet);
}

int nwk_config_add_mask (uint16_t network, uint8_t length, char device)
{
#ifdef NWK_DISABLE_SEND_REMOTE_CONFIG
	(void) network;
	(void) length;
	(void) device;
	return -ENOSYS;
#else
	if (length > 16) return -EINVAL;
	if (network & ((1 << (16 - length)) - 1)) return -EINVAL;
	if (outgoing_packet.length > max_packet_length - 4) return -EMSGSIZE;
	uint8_t *buf = (uint8_t *) outgoing_packet.data;
	buf[outgoing_packet.length++] = device;
	buf[outgoing_packet.length++] = length;
	buf[outgoing_packet.length++] = (uint8_t) (network >> 8);
	buf[outgoing_packet.length++] = (uint8_t) (network & 0xFF);
	return 0;
#endif
}

int nwk_config_add_span (uint16_t first, uint16_t last, char device)
{
#ifdef NWK_DISABLE_SEND_REMOTE_CONFIG
	(void) first;
	(void) last;
	(void) device;
	return -ENOSYS;
#else
	if (last < first) return -EINVAL;
	uint16_t span = last - first;
	uint8_t *buf = (uint8_t *) outgoing_packet.data;
	if (span < 239) {
		if (outgoing_packet.length > max_packet_length - 4) return -EMSGSIZE;
		buf[outgoing_packet.length++] = device;
		buf[outgoing_packet.length++] = span + 16;
	} else {
		if (outgoing_packet.length > max_packet_length - 6) return -EMSGSIZE;
		buf[outgoing_packet.length++] = device;
		buf[outgoing_packet.length++] = 0xFF;
		buf[outgoing_packet.length++] = (uint8_t) (span >> 8);
		buf[outgoing_packet.length++] = (uint8_t) (span & 0xFF);
	}
	buf[outgoing_packet.length++] = (uint8_t) (first >> 8);
	buf[outgoing_packet.length++] = (uint8_t) (first & 0xFF);
	return 0;
#endif
}
