//=========================================================================
// main.c
// Example program illustrating usage of the ToLHnet protocol.
// Developed for a Texas Instruments Tiva C Series TM4C123G
// LaunchPad Evaluation Board (EK-TM4C123GXL)
/*=========================================================================
Copyright (C) 2014 Giorgio Biagetti.

Distributed under the Boost Software License, Version 1.0.

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

=========================================================================*/

#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>
#include <inc/hw_memmap.h>
#include <inc/hw_types.h>
#include <inc/hw_gpio.h>
#include <inc/hw_ints.h>
#include <inc/hw_sysctl.h>
#include <driverlib/gpio.h>
#include <driverlib/pin_map.h>
#include <driverlib/sysctl.h>
#include <driverlib/uart.h>
#include <driverlib/udma.h>
#include <driverlib/systick.h>
#include <driverlib/interrupt.h>
#include <driverlib/rom.h>

#include "Drivers/uartdma.h"
#include "DataLink/UART/uart_mac.h"
#include "DataLink/PLC/plc_mac.h"
#include "DataLink/W433/w433_mac.h"
#include "Application/application.h"

#include "board.h"
#include "serial.h"

uint8_t DMAControlTable[1024] __attribute__ ((used,aligned(1024)));


// Keyboard driver:
/* keyboard status flags - bits numbered as in GPIO_INT_PIN_x */
#define KEY_ENABLED_BUTTONS ( GPIO_INT_PIN_0 | GPIO_INT_PIN_4 )
#define KEY_GPIO_PORT GPIO_PORTF_BASE

static volatile uint32_t active_buttons;
static uint32_t fall_debounce;
static uint32_t rise_debounce;

static volatile bool timer_needed;

void isr_keyboard (void)
{
	uint32_t int_status = HWREG(KEY_GPIO_PORT + GPIO_O_RIS);
	uint32_t key_status = ~int_status | ~KEY_ENABLED_BUTTONS;
	/* compute just-pressed button mask */
	uint32_t pressed_buttons = ~(key_status | active_buttons);
	/* record button state */
	active_buttons |= pressed_buttons;
	fall_debounce |= pressed_buttons;
	/* Numb the port to debounce the key */
	HWREG(KEY_GPIO_PORT + GPIO_O_IM) &= ~pressed_buttons;
	/* acknowledge interrupt */
	HWREG(KEY_GPIO_PORT + GPIO_O_ICR) = int_status;
}

void tmr_keyboard (void)
{
	uint32_t key_status = HWREG(KEY_GPIO_PORT + GPIO_O_DATA + (KEY_ENABLED_BUTTONS << 2)) | ~KEY_ENABLED_BUTTONS;
	/* compute just-released button mask */
	uint8_t released_buttons = active_buttons & ~fall_debounce & key_status;
	/* re-enable interrupts after debouncing */
	HWREG(KEY_GPIO_PORT + GPIO_O_ICR) = rise_debounce;
	HWREG(KEY_GPIO_PORT + GPIO_O_IM) |= rise_debounce;
	fall_debounce = 0;
	/* debounce newly released buttons */
	rise_debounce = released_buttons;
	/* update active button mask */
	active_buttons &= ~released_buttons;
	/* acknowledge interrupt */
	//TODO
	/* possibly stop timer */
	if (!timer_needed) {} // TODO
}

void key_event (int button);

void key_process (void)
{
	static uint32_t prev_button_state = 0;
	/* Wait for something to happen... */
	/* active_buttons may change at unexpected times, make a copy! */
	uint32_t curr_button_state = active_buttons; // TODO: check atomicity!
	/* process button state changes: */
	if (prev_button_state != curr_button_state) {
		uint32_t event = ~prev_button_state & curr_button_state;
		// process the event:
		if (event & GPIO_INT_PIN_0) key_event(2); // SW2 pressed
		if (event & GPIO_INT_PIN_4) key_event(1); // SW1 pressed
		if (curr_button_state && !prev_button_state) {
			/* Start the debounce timer */
			timer_needed = true;
		} else if (!curr_button_state && prev_button_state) {
			/* Stop the debounce timer */
			timer_needed = false;
			/* Put off actual deactivation until the proper ISR decides to do so,
				we need an extra timer pulse for debouncing! */
		}
		prev_button_state = curr_button_state;
	}
}

void key_event (int button)
{
	static const uint8_t msg1[] = { 0x01, 0x01 };
	static const uint8_t msg2[] = { 0x01, 0x02 };
	if (button == 1) {
// 		BOARD_LED_RED = 0;
// 		BOARD_LED_GREEN = 0;
// 		BOARD_LED_BLUE = 0;
		send_datagram(0, msg1, sizeof msg1, CODE_MSG);
	}
	if (button == 2) {
//		HWREG(BITBAND(&w433_event, 2)) = 1;
		send_datagram(0, msg2, sizeof msg2, CODE_MSG);
	}
}


#define SYSTICKS_PER_SECOND 100
static volatile uint32_t ticks;

void SysTickIntHandler (void)
{
	++ticks;
	uint32_t seconds = ticks / 100;
	tmr_keyboard();
}

void delay_cs (uint32_t cs)
{
	uint32_t now = ticks;
	while (ticks - now < cs) ;
}

char msg[] = "ToLHnet firmware v0.1 ready!\r\n";

__attribute__ ((section(".mac_address"))) static const volatile uint64_t mac = 0xABCDEF000000;

int main (void)
{
	// setup the system clock to run at 80 MHz from the external crystal:
	ROM_SysCtlClockSet(SYSCTL_SYSDIV_2_5 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN | SYSCTL_XTAL_16MHZ);
	// setup the system tick:
	ROM_SysTickPeriodSet(ROM_SysCtlClockGet() / SYSTICKS_PER_SECOND);
	ROM_SysTickIntEnable();
	ROM_SysTickEnable();
	// enable peripherals to operate when CPU is in sleep:
	ROM_SysCtlPeripheralClockGating(true);

	// enable the uDMA controller at the system level:
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_UDMA);
	ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_UDMA);
	ROM_uDMAEnable();
	ROM_uDMAControlBaseSet(DMAControlTable);

	// enable the uDMA controller error interrupt:
//	ROM_IntEnable(INT_UDMAERR);

	// enable all of the GPIOs:
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);
	ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_GPIOA);
	ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_GPIOB);
	ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_GPIOC);
	ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_GPIOD);
	ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_GPIOE);
	ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_GPIOF);

	// unlock commit registers for PF0 & PD7 (NMI pins):
	HWREG(GPIO_PORTD_BASE + GPIO_O_LOCK) = 0x4C4F434B;
	HWREG(GPIO_PORTF_BASE + GPIO_O_LOCK) = 0x4C4F434B;
	HWREG(GPIO_PORTD_BASE + GPIO_O_CR)   = 0xFF;
	HWREG(GPIO_PORTF_BASE + GPIO_O_CR)   = 0xFF;
	HWREG(GPIO_PORTD_BASE + GPIO_O_LOCK) = 0;
	HWREG(GPIO_PORTF_BASE + GPIO_O_LOCK) = 0;

	// setup pins connected to RGB LED:
	ROM_GPIOPinTypeGPIOOutput(GPIO_PORTF_BASE, GPIO_PIN_1 | GPIO_PIN_2 | GPIO_PIN_3);
	ROM_GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_1 | GPIO_PIN_2 | GPIO_PIN_3, 0);

	// setup pins connected to pushbuttons:
	ROM_GPIOPinTypeGPIOInput(GPIO_PORTF_BASE, GPIO_PIN_0 | GPIO_PIN_4);
	ROM_GPIOPadConfigSet(GPIO_PORTF_BASE, GPIO_PIN_0 | GPIO_PIN_4, 0, GPIO_PIN_TYPE_STD_WPU);
	ROM_IntEnable(INT_GPIOF);
	GPIOIntTypeSet(GPIO_PORTF_BASE, GPIO_PIN_0 | GPIO_PIN_4, GPIO_FALLING_EDGE);
	GPIOIntEnable(GPIO_PORTF_BASE, KEY_ENABLED_BUTTONS);

//	delay_cs(100);

	// setup application layer:
	app_init();

	// setup network layer:
	nwk_init(mac);
	uart_mac_init();

	uart_send_frame(0, msg, sizeof msg - 1); //TODO: remove me!
	serial_wait();
	
	plc_mac_init();
	w433_mac_init();
	datagram_received = data_received;
	netevent_received = network_event_handler;

	// main loop:
	while (1) {
//		BOARD_LED_BLUE = 0;
		ROM_SysCtlSleep();
//		BOARD_LED_BLUE = 1;
		key_process();
		if (w433_event) {
			w433_process_event();
		}
		if (uart_data_available[0]) {
			uart_receive_packet(0, uart_rx_buffer[0], uart_data_length[0]);
			uart_data_available[0] = false;
		}
		if (uart_data_available[1]) {
			uart_receive_packet(1, uart_rx_buffer[1], uart_data_length[1]);
			uart_data_available[1] = false;
		}
		if (uart_data_available[2]) {
			plc_receive_packet(2, uart_rx_buffer[2], uart_data_length[2]);
			uart_data_available[2] = false;
		}
		if (uart_data_available[3]) {
			uart_receive_packet(3, uart_rx_buffer[3], uart_data_length[3]);
			uart_data_available[3] = false;
		}

	}
}
